--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Tip System
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Tip class definition
LTip =
{
	__tostring = function(t) return "{Tip:"..tostring(t.first).."/"..tostring(t.last).."-"..tostring(t.type).."-"..tostring(t.target).."-"..tostring(t.port).."-"..tostring(t.event).."}" end,
	
	_Active = {},
}

function LTip:new(t)
	t = t or {} setmetatable(t, self) self.__index = self
	t:Initialize()
	table.insert(self._Active, t)
	return t
end

-------------------------------------------------------------------------------
-- Reset

function LTip:ResetAll()
	self._Active = {}
end

-------------------------------------------------------------------------------
-- Load and Save

function LTip:BuildOneSaveTable()
	local t = {}
	for key,val in pairs(self) do
		if type(val) ~= "function" then
			t[key] = val
		end
	end
	return t
end

function LTip:BuildSaveTable()
	local t = {}
	for _,tip in pairs(self._Active) do
		local one = tip:BuildOneSaveTable()
		table.insert(t, one)
	end
	return t
end

function LTip:LoadSaveTable(t)
	if t then
		for _,val in ipairs(t) do
			val = LTip:new(val)
		end
	end
end

-------------------------------------------------------------------------------
-- Messaging

function LTip:GetMessage()
	return GetString(self.event, GetString(self.target), GetString(self.port))
end

-------------------------------------------------------------------------------
-- Application Functions

local function ApplyPriceChange(tip)
	if tip.first <= gSim.weeks and gSim.weeks <= tip.last then
		if gSim.port.name == tip.port then
			local item = LItem:ByName(tip.target)
			if item then
				-- 20% bump up or down, capped at low or high
				if tip.type == "up" then
					item.price = bsgFloor(item.price * 1.2 + .5)
					if item.price > item.high then item.price = item.high end
				else
					item.price = bsgFloor(item.price * .8 + .5)
					if item.price < item.low then item.price = item.low end
				end
			end
		end
		return true
	else
		return false
	end
end

-------------------------------------------------------------------------------
-- Initialization

function LTip:Initialize()
	self.Apply = ApplyPriceChange
end

-------------------------------------------------------------------------------
-- Tip Application

function LTip:ApplyAll()
	local n = 1
	while n <= table.getn(self._Active) do
		local tip = self._Active[n]
		if tip:Apply() then
			n = n + 1
		else
			-- Tip expired
			table.remove(self._Active, n)
		end
	end
end

-------------------------------------------------------------------------------
-- Tip Randomization

local ingUpEventList = { "ev_fire","ev_strike","ev_ing_priceup01","ev_ing_priceup02","ev_ing_priceup03","ev_ing_priceup04","ev_ing_priceup05" }
local ingDownEventList = { "ev_glut","ev_ing_pricedown01","ev_ing_pricedown02","ev_ing_pricedown03","ev_ing_pricedown04","ev_ing_pricedown05" }

local prodUpEventList = { "ev_demand","ev_prod_priceup01","ev_prod_priceup02","ev_prod_priceup03","ev_prod_priceup04","ev_prod_priceup05", }
local prodDownEventList = { "ev_tired","ev_prod_pricedown01","ev_prod_pricedown02","ev_prod_pricedown03","ev_prod_pricedown04","ev_prod_pricedown05" }

local function RandomMarketPort()
	-- Pick a location from all available ports with markets
	local ports = {}
	for port in LPort:AllPorts() do
		if port ~= gSim.port and port.available and port.hasMarket and (not port.hidden) then
			table.insert(ports, port)
		end
	end
	
	local p = nil
	if table.getn(ports) > 0 then
		local i = bsgRandom(1, table.getn(ports))
		p = ports[i]
	end
	if not p then p = LPort:ByName("sanfrancisco") end		-- worst case use SF
	return p
end

local function RandomShopPort()
	-- Pick a location from all available ports with shops
	local ports = {}
	for port in LPort:AllPorts() do
		if port ~= gSim.port and port.available and port.hasShop then
			table.insert(ports, port)
		end
	end
	
	local p = nil
	if table.getn(ports) > 0 then
		local i = bsgRandom(1, table.getn(ports))
		p = ports[i]
	end
	if not p then p = LPort:ByName("sanfrancisco") end		-- worst case use SF
	return p
end

local function RandomMadeRecipeName()
	-- Pick a product from all manufactured recipes
	local products = {}
	for p in LItem:KnownRecipes() do
		if p.inventory > 0 or p.sold > 0 then
			table.insert(products, p.name)
		end
	end
	
	local p = nil
	if table.getn(products) > 0 then
		local i = bsgRandom(1, table.getn(products))	
		p = products[i]
	end
	if not p then p = "b_01" end		-- worst case use basic squares
	return p
end

local function RandomFactoryRecipeName()
	-- Pick a product from inventory
	local products = {}
	for f in LFactory:AllFactories() do
		if f.owned and f.product and (f.product.inventory > 0 or f.product.sold > 0) then
			table.insert(products, f.product.name)
		end
	end
	
	local p = nil
	if table.getn(products) > 0 then
		local i = bsgRandom(1, table.getn(products))
		p = products[i]
	end
	if not p then p = RandomMadeRecipeName() end		-- worst case use something made (basic squares default)
	return p
end

local function RandomIngredientName(port)
	-- Pick an ingredient from all ingredients available in markets at this port
	local ingredients = {}
	for _,m in ipairs(LMarket._Markets) do
		if m.port == port then
			-- Will double up if two markets sell the same ingredient
			-- Don't give tips on seasonal ingredients
			for ing,_ in pairs(m.inventory) do
				local item = LItem:ByName(ing)
				if not item:IsSeasonal() then table.insert(ingredients, ing) end
			end
		end
	end

	local p = nil
	if table.getn(ingredients) > 0 then
		local i = bsgRandom(1, table.getn(ingredients))
		p = ingredients[i]
	end
	if not p then p = "sugar" end		-- worst case use sugar
	return p
end

function LTip:RandomTip(ingprod, updown)
	local tip = {}
	
	-- Decide up/down
	local ingEvents,prodEvents
	local rand = bsgRandom(1,100)
	if rand > 50 or updown == "up" then
		tip.type = "up"
		ingEvents = ingUpEventList
		prodEvents = prodUpEventList
	else
		tip.type = "down"
		ingEvents = ingDownEventList
		prodEvents = prodDownEventList
	end
	
	-- Decide a target product or ingredient and a random reason
	rand = bsgRandom(1,100)
	if ingprod == "prod" then
		-- Product currently in manufacture
		tip.port = RandomShopPort()
		tip.target = RandomFactoryRecipeName()
		tip.event = prodEvents
	elseif rand > 50 then
		-- Product
		tip.port = RandomShopPort()
		tip.target = RandomMadeRecipeName()
		tip.event = prodEvents
	else
		-- Ingredient
		tip.port = RandomMarketPort()
		tip.target = RandomIngredientName(tip.port)
		tip.event = ingEvents
	end
	
	tip.port = tip.port.name
	
	local i = bsgRandom(1, table.getn(tip.event))
	tip.event = tip.event[i]
	
	tip.first = gSim.weeks
	tip.last = gSim.weeks + 10
	
	return LTip:new(tip)
end
